/*

BSD License

Copyright (c) 2002, Brent Simmons
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

*	Redistributions of source code must retain the above copyright notice,
	this list of conditions and the following disclaimer.
*	Redistributions in binary form must reproduce the above copyright notice,
	this list of conditions and the following disclaimer in the documentation
	and/or other materials provided with the distribution.
*	Neither the name of ranchero.com or Brent Simmons nor the names of its
	contributors may be used to endorse or promote products derived
	from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


*/

/*
	Huevos.m
	Huevos

	Created by Brent Simmons on Thu Mar 28 2002.
	Copyright (c) 2002 Brent Simmons. All rights reserved.
*/


#import "Huevos.h"
#import "SearchEngines.h"
#import "HotKeyPrefs.h"


#define hotKeyIdentifier 'HHok'


static id myInstance = nil;


@implementation Huevos


- (void) awakeFromNib {
	
	/*
	Startup tasks.	
	*/
	
	NSNotificationCenter *nc = [NSNotificationCenter defaultCenter];;
	
	myInstance = self;
	
	[self setupPrefs];
	
	[self updatePopupMenu];
	
	/*Register for getting notifications when
	various prefs change.*/
	
	[nc addObserver: self
		selector: @selector (handleSearchEnginesChange:)
		name: @"SearchEnginesChanged"
		object: nil];
	
	[nc addObserver: self
		selector: @selector (handleHotKeyChange:)
		name: @"HotKeyChanged"
		object: nil];

	[self getFloatingPref];
	
	[self setFloatingMenuItemState];

	[self setWindowFloats];
	
	[self rememberLastSearch];
	
	[self setupHotKey]; /*Register our hot key for bringing Huevos to the front*/
	} /*awakeFromNib*/


+ (id) sharedInstance {
	
	return (myInstance);
	} /*sharedInstance*/
	
	
/*Actions*/

- (IBAction) openMainWindow: (id) sender {

	[mainWindow makeKeyAndOrderFront: self];	
	} /*openMainWindow*/


- (IBAction) doSearch: (id) sender {
	
	/*
	Run a search in the browser. The name of the search engine
	comes from the popup menu. The query string comes from the
	input text field. The URL to use comes from prefs; it's
	associated with the name of the search engine.
	
	PBS 02/18/03: two bug fixes: high-ASCII characters get
	encoded properly now; spaces now get turned into + characters.
	*/

	NSString *baseUrl;
	NSString *searchUrl, *encodedSearchUrl;	
	NSURL *url;	
	
	/*Create a string version of the URL to send to the browser.*/
	
	baseUrl = [SearchEngines urlFromName: [popupButton titleOfSelectedItem]];
	
	searchUrl = [NSString stringWithFormat: baseUrl, [searchField stringValue]];
	
	/*URL-encode the URL string. Call a CoreFoundation routine. Apparently
	there's no Cocoa routine for URL-encoding (surprisingly).*/
	
	encodedSearchUrl = (NSString *) CFURLCreateStringByAddingPercentEscapes
		(NULL, (CFStringRef) searchUrl, CFSTR(" "), NULL, kCFStringEncodingWindowsLatin1);
	
	encodedSearchUrl = [self replaceAll: @" " with: @"+" inString: encodedSearchUrl];

	/*Create the URL object.*/
	
	url = [NSURL URLWithString: encodedSearchUrl];

	/*Open the URL in the default browser.*/
	
	[[NSWorkspace sharedWorkspace] openURL: url];
	
	[self saveLastSearchParams];
	} /*doSearch*/


- (IBAction) setFloating: (id) sender {
	
	if (useFloatingPanel) /*toggle*/	
		useFloatingPanel = NO;
	else
		useFloatingPanel = YES;
	
	[self setFloatingMenuItemState];
	
	[self setFloatingPref];
	
	[self setWindowFloats];
	} /*setFloating*/


- (IBAction) openHelpInBrowser: (id) sender {
	
	/*
	Instead of using the less-than-fun Apple Help app,
	the Help page is a page on the ranchero.com site.
	*/
	
	NSURL *url = [NSURL URLWithString: @"http://ranchero.com/huevos/help/huevos11.php"];
	
	[[NSWorkspace sharedWorkspace] openURL: url];
	} /*openHelpInBrowser*/
	
	
/*String utility*/

- (NSString *) replaceAll: (NSString *) searchFor with: (NSString *) replaceWith
	inString: (NSString *) s {
	
	NSArray *stringComponents = [s componentsSeparatedByString: searchFor];

	return [stringComponents componentsJoinedByString: replaceWith];
	} /*replaceAll*/


/*UI stuff*/

- (void) setFloatingMenuItemState {
	
	if (useFloatingPanel)
		[floatMenuItem setState: NSOnState];		
	else
		[floatMenuItem setState: NSOffState];
	} /*setFloatingMenuItemState*/


- (void) updatePopupMenu {
	
	NSMenu *menu;
	int ct, i;
	
	[popupButton removeAllItems]; /*Clear the popup menu.*/
	
	[popupButton addItemsWithTitles: [SearchEngines arrayOfNames]]; /*Add items to the popup menu.*/
	
	/*Build the menu*/
	
	menu = [popupButton menu];
	ct = [menu numberOfItems];
	
	for (i = 0; i < ct; i++) {
		
		NSString *commandKey = [SearchEngines nthSearchEngineCommandKey: i];
		NSMenuItem *menuItem;
		
		if (commandKey == nil)
			continue;
		
		if ([commandKey isEqualToString: @""])
			continue;
	
		menuItem = [menu itemAtIndex: i];
	
		[menuItem setKeyEquivalent: commandKey];
		} /*for*/
	} /*updatePopupMenu*/


- (void) setWindowFloats {
	
	[mainWindow setFloatingPanel: useFloatingPanel];	
	} /*setWindowFloats*/


- (void) goUpOneSearchEngine {
	
	/*
	Select the previous search engine in the popup menu.
	Does not wrap around (on purpose).
	*/
	
	int current = [popupButton indexOfSelectedItem];
	int new = current - 1;
	
	if (new < 0)
		return;
		//uncomment to wrap around: new = [popupButton numberOfItems] - 1;
	
	[popupButton selectItemAtIndex: new];
	} /*goUpOneSearchEngine*/
	

- (void) goDownOneSearchEngine {
	
	/*
	Select the next search engine in the popup menu.
	Does not wrap around (on purpose).
	*/
	
	int current = [popupButton indexOfSelectedItem];
	int new = current + 1;
	
	if (new >= [popupButton numberOfItems])		
		new = [popupButton numberOfItems] - 1;
		//uncomment to wrap around: new = 0;
	
	[popupButton selectItemAtIndex: new];	
	} /*goDownOneSearchEngine*/


- (BOOL) validateMenuItem: (NSMenuItem *) menuItem {
	
	if (menuItem == newMenuItem) {
		
		/*Disable the New item if the search window is on screen.*/
		
		if ([mainWindow isVisible])
			return (NO);		
		} /*if*/
	
	return (YES);
	} /*validateMenuItem*/
	
	
/*HotKey prefs*/

- (int) hotkeyModifierCode {
	
	/*
	Get the hot key modifier from prefs. Translate the prefs representation
	to the system representation.
	*/
	
	int val = [HotKeyPrefs hotKeyModifier];
	
	if (val == controlKeyPref)
		return (controlKey);
	
	if (val == optionKeyPref)
		return (optionKey);
	
	if (val == commandKeyPref)
		return (cmdKey);
	
	return (0);
	} /*hotkeyModifierCode*/
	

- (int) hotKeyCode {
	
	/*
	Get the hot key code from prefs. Translate the prefs representation
	to the system representation.
	*/
	
	int val = [HotKeyPrefs hotKey];

	switch (val) {
		
		case escapeKeyPref:
			return (53);
		
		case f1KeyPref:
			return (122);
		
		case f2KeyPref:
			return (120);
		
		case f3KeyPref:
			return (99);
		
		case f4KeyPref:
			return (118);
		
		case f5KeyPref:
			return (96);
		
		case f6KeyPref:
			return (97);
		
		case f7KeyPref:
			return (98);
		
		case f8KeyPref:
			return (100);
		
		case f9KeyPref:
			return (101);
		
		case f10KeyPref:
			return (109);
		
		case f11KeyPref:
			return (103);
		
		case f12KeyPref:
			return (111);		
		} /*switch*/
		
	return (0); /*Function key*/
	} /*hotKeyCode*/

	
static EventHotKeyRef hotKeyRef = nil;

- (void) setupHotKey {
	
	/*
	Register the hot key to bring the app to the front. First unregister the previous
	hot key if there was one.
	*/
	
	EventHotKeyID hotKeyID;
	OSErr err = noErr;

	if (hotKeyRef != nil) /*Unregister old hot key*/
		UnregisterEventHotKey (hotKeyRef);
	
	/*Register new hot key*/

	hotKeyID.id = 1;
	
	hotKeyID.signature = hotKeyIdentifier;
	
	err = RegisterEventHotKey ([self hotKeyCode], [self hotkeyModifierCode], hotKeyID,
		GetApplicationEventTarget (), 0, &hotKeyRef);
	
	if (err != noErr)
		NSLog (@"Error registering hot key: %d", err);
	} /*setupHotKey*/
	

/*Notifications*/

- (void) handleSearchEnginesChange: (NSNotification *) note {
	
	/*
	The list of search engines changed. Update the popup menu.
	*/
	
	[self updatePopupMenu];
	} /*handleSearchEnginesChanged*/
	

- (void) handleHotKeyChange: (NSNotification *) note {
	
	/*
	Hot key changed. Re-register it.
	*/
	
	[self setupHotKey];	
	} /*handleHotKeyChange*/
	

- (BOOL) applicationShouldHandleReopen: (NSApplication *) app hasVisibleWindows: (BOOL) fl {
	
	/*
	Icon was clicked in the Dock. Open main window if necessary.
	*/
	
	if (!fl) {
	
		[self openMainWindow: nil];
		
		return (NO);
		} /*if*/
	
	return (YES);
	} /*applicationShouldHandleReopen*/
	

/*General prefs*/

- (void) rememberLastSearch {
	
	NSString *searchString, *searchEngine;
	NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
	
	searchString = [defaults objectForKey: lastSearchStringPrefsKey];
	
	searchEngine = [defaults objectForKey: lastSearchEnginePrefsKey];
	
	if (searchEngine != nil)
		[popupButton selectItemWithTitle: searchEngine];
	
	if (searchString != nil)	
		[searchField setStringValue: searchString];
	} /*rememberLastSearch*/


- (void) saveLastSearchParams {
	
	NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
	
	[defaults setObject: [popupButton titleOfSelectedItem] forKey: lastSearchEnginePrefsKey];

	[defaults setObject: [searchField stringValue] forKey: lastSearchStringPrefsKey];
	} /*saveLastSearchParams*/
	

- (void) getFloatingPref {
		
	useFloatingPanel = [[NSUserDefaults standardUserDefaults] boolForKey: useFloatingPanelPrefsKey];	
	} /*getFloatingPref*/
	

- (void) setFloatingPref {
	
	[[NSUserDefaults standardUserDefaults] setBool: useFloatingPanel forKey: useFloatingPanelPrefsKey];
	} /*setFloatingPref*/


- (void) setupPrefs {
	
	/*
	Register the default search engines in prefs storage.
	*/
	
	NSMutableDictionary *defaultValues = [NSMutableDictionary dictionary];
	NSDictionary *defaultSearchEngines, *defaultCommandKeys;
	NSString *path;
	
	/*Get default search engines from plist file on disk.*/
	
	path = [[NSBundle mainBundle] pathForResource: defaultSearchEnginesFileName ofType: @"plist"];
	
	defaultSearchEngines = [NSDictionary dictionaryWithContentsOfFile: path];
	
	/*Put default search engines in prefs storage as a sub-dictionary.*/
	
	[defaultValues setObject: defaultSearchEngines forKey: searchEnginesPrefsKey];
	
	[defaultValues setObject: defaultSearchEngineName forKey: lastSearchEnginePrefsKey];
	
	/*Command keys*/
		
	path = [[NSBundle mainBundle] pathForResource: defaultCommandKeysFileName ofType: @"plist"];

	defaultCommandKeys = [NSDictionary dictionaryWithContentsOfFile: path];
	
	[defaultValues setObject: defaultCommandKeys forKey: commandKeysPrefsKey];
	
	/*Default hot key prefs.*/
	
	[defaultValues setObject: [NSNumber numberWithInt: controlKeyPref] forKey: hotKeyModifierPrefsKey];
	
	[defaultValues setObject: [NSNumber numberWithInt: escapeKeyPref] forKey: hotKeyPrefsKey];
	
	/*Register default prefs.*/
	
	[[NSUserDefaults standardUserDefaults] registerDefaults: defaultValues];
	} /*initialize*/


@end